/*
 * Real-time simulation of the OU Houvercraft.
 * 
 * This is intended as a drop-in "replacement" for the physical system.
 * 
 * Author: Andrew H. Fagg   Symbiotic Computing Laboratory 2020-04-15
 * 
 */

#include "Houvercraft.h"
#include "Stream.h"
#include <Arduino.h>

/**
 * Constructor
 * 
 * @param sensor0_dir Direction relative to front of the 0th distance sensor (in degrees; left-handed)
 * @param sensor1_dir Direction relative to front of the 1st distance sensor 
 */
Houvercraft::Houvercraft(float sensor0_dir, float sensor1_dir)
{
  this->distance_sensor_directions[0] = DEG2RAD(sensor0_dir);
  this->distance_sensor_directions[1] = DEG2RAD(sensor1_dir);
  this->stochastic = 1;
  this->debug_flag = 0;
  this->task = 0;
  
  this->reset();
};

/**
 * Reset the hovercraft to a known state
 * 
 * For task 0: X/Y position is 0.5, 0.5 (with orientation 0)
 * For task 1: X/Y position can be either 0.5,0.5 or 0.5, 3.5
 */
void Houvercraft::reset()
{
  // Set velocities and accumulated position to zeros
  for(int i = 0; i < this->N_DIM; ++i) {
    this->velocity[i] = this->accumulated_position[i] = 0.0;
  }

  // Initialize position based on task
  if(this->task == 0) {
    // Simple task
    this->position[0] = 0.5;
    this->position[1] = 0.5;
    this->position[2] = 0.0;
  }else if(this->task == 1)
  {
    // Randomly pick one of two start conditions
    if(rand()%2 == 0) {
      this->position[0] = 0.5;
      this->position[1] = 0.5;
      this->position[2] = 0.0;
    }else{
      this->position[0] = 0.5;
      this->position[1] = 3.5;
      this->position[2] = 0.0;
    }
  }

  // Central fan to off
  this->central_fan = 0.0;

  // Fan magnitudes to zero
  for(int i = 0; i < this->N_FANS; ++i) {
    this->fan_magnitudes[i] = 0.0;
  }

  // Collision count
  this->collisions = 0;

};

/**
 * Set the current task for the environment.  This is used by reset()
 * 
 * @param t New task (either 0 or 1)
 * 
 */
void Houvercraft::set_task(int t)
{
  this->task = t;
}

/**
 * Set the magnitudes of the lateral fans 
 * 
 * @param magnitudes Array of magnitudes (left, right, rear)
 */
void Houvercraft::set_lateral_fan_magnitudes(float magnitudes[N_FANS])
{
  for(int i = 0; i < this->N_FANS; ++i) {
    this->fan_magnitudes[i] = magnitudes[i];
  }
}

/**
 * Print the true state of the houvercraft
 */
void Houvercraft::print_state()
{
  Serial.printf("X = %0.04f (%0.04f) \tY = %0.04f (%0.04f) \ttheta = %0.04f (%0.04f)\tcollisions = %2d\t", 
    (this->position[0]), (this->velocity[0]), 
    (this->position[1]), (this->velocity[1]), 
    RAD2DEG(this->position[2]), RAD2DEG(this->velocity[2]),
    this->collisions);
}

/**
 * Return the rotational velocity.  
 * 
 * If we are in stochastic mode, then some noise is added to the measurement.
 * 
 * @return The rotational velocity in degrees/s (left-handed coordinate frame)
 */
float Houvercraft::imu_gz()
{
  float vel = RAD2DEG(this->velocity[DIM_THETA]);
  if(this->stochastic)
  {
    vel += RAND_GZ * get_rand();
  }
  return(vel);
}

/**
 * Return the compass heading
 * 
 * If we are in stochastic mode, then some noise is added to the measurement.
 * 
 * @return The compass heading in degrees (left-handed coordinate frame)
 */
float Houvercraft::imu_yaw()
{
  float deg = RAD2DEG(this->position[DIM_THETA]);
  if(this->stochastic)
  {
    deg += RAND_YAW * get_rand();
  }
  return(deg);
}

/**
 * Set the magnitude of the central fan
 * 
 * @param mag The magnitude of the fan.  For the simulation, it only matters that the magnitude is 
 *   to the left or right of FAN_THRESHOLD
 */
void Houvercraft::fan_write(int mag)
{
  this->central_fan = mag;
}

/**
 * Returns the change in Cartesian position since the last call
 * 
 * @param cartesian_change Array of positional change values (x, y, theta in m,m,deg)
 * 
 * If we are in stochastic mode, then some noise is added to the measurement.
 * 
 */
void Houvercraft::compute_chassis_motion(float cartesian_change[N_DIM])
{
  // Copy accumulated values; zero out accumulation
  for(int i = 0; i < N_DIM; ++i)
  {
    cartesian_change[i] = accumulated_position[i];
    accumulated_position[i] = 0.0;  
    
  }

  // Change units
  cartesian_change[2] = RAD2DEG(cartesian_change[2]);
  
  if(this->stochastic && (this->central_fan > FAN_THRESHOLD))
  {
    cartesian_change[0] += RAND_CART * get_rand();
    cartesian_change[1] += RAND_CART * get_rand();
    cartesian_change[2] += RAND_ROT * get_rand();
  }
}

/**
 * Set the stochastic mode
 * 
 * @param stochastic mode (0 = none, 1 = stochastic)
 * 
 */
void Houvercraft::set_stochastic(int stochastic)
{
  this->stochastic = stochastic;
}

/**
 * Set debug mode
 * 
 * @param debug mode
 */
void Houvercraft::set_debug(int debug)
{
  this->debug_flag = debug;
}

/*
 * Return a uniformally distributed float in the range -0.5 to 0.5
 * 
 */
float Houvercraft::get_rand()
{
  return(((float) rand()/(float)RAND_MAX)-0.5);
}

/**
 * Print out the environmental map with the houvercraft location indicated
 */
void Houvercraft::print_map()
{
  // Which marker to use?
  char c;
  if(central_fan > FAN_THRESHOLD)
    c = '@';
  else
    c = 'o';

  // Generate the printed map
  this->map.print(this->position, c);
}

/**
 * Read from the specified distance sensor.  Note that distances are measured from the
 * center of the craft.
 * 
 * @param side One of SHARP_0, SHARP_1, corresponding to the sensor directions
 *   specified in the constructor
 * @return Distance in cm to the nearest wall
 * 
 */
float Houvercraft::read_distance(SharpSensor side)
{
  // Convert from m to cm
  float d = (this->map.intersection_distance(this->position, 
      this->position[2] + this->distance_sensor_directions[side]));

  if(this->stochastic && d < Segment::MAX_DISTANCE)
  {
    // Add stochastic value
    d = d + RAND_DISTANCE * get_rand();

    // Correct if outside viable range
    if(d < 0.0)
      d = 0.0;
    else if(d > Segment::MAX_DISTANCE)
      d = Segment::MAX_DISTANCE;
  }

  // Convert to cm
  return(100.0 * d);
}

/**
 * Set the true position of the craft
 * 
 * @param x coordinate (m)
 * @param y coordinate (m)
 * @param theta orientation in degrees (left-handed coordinate frame)
 */
void Houvercraft::set_position(float x, float y, float theta)
{
  this->position[0] = x;
  this->position[1] = y;

  // correct the range
  while(theta > 180.0)
     theta = theta - 360.0;
  while(theta < -180.0)
     theta = theta + 360.0;
     
  this->position[2] = DEG2RAD(theta);
}

/**
 * Take one simulation step
 * 
 * Expects to be called once per 5ms
 * 
 */
void Houvercraft::step()
{
  // Compute net force and torque
  const float c30 = cos(DEG2RAD(30));
  const float s30 = sin(DEG2RAD(30));
  float fx, fy, torque;
  float ax, ay, a_theta;
  float dv[N_DIM];
  float dvg[N_DIM];

  // Are we flying?
  if(this->central_fan > FAN_THRESHOLD) {
    // Forces and torques in the houvercraft coordinate frame
    fx = c30 * this->fan_magnitudes[0] + c30 * this->fan_magnitudes[1];
    fy = -s30 * this->fan_magnitudes[0] + s30 * this->fan_magnitudes[1] - this->fan_magnitudes[2];
    torque = RADIUS * (- this->fan_magnitudes[0] + this->fan_magnitudes[1] + this->fan_magnitudes[2]);

    // Add some random elements to the forces?
    if(this->stochastic)
    {
      float r = RAND_FORCE * get_rand();
      fx += r;

      r = RAND_FORCE * get_rand();
      fy += r;

      r = RAND_TORQUE * get_rand();
      torque += r;
    }
    
    //if(this->debug_flag == 1)
    //  Serial.printf("F: %3.5f %3.5f %3.5f\n", fx, fy, torque);
      
    // Acceleration: local coord frame
    ax = fx / MASS;
    ay = fy / MASS;
    a_theta = torque / R_INERTIA;

    // Orientation: local & global coord frames
    this->velocity[2] += a_theta * DT;
    this->position[2] += this->velocity[2] * DT;
    this->accumulated_position[2] += this->velocity[2] * DT;

     // Deal with wrap-around
    if(this->position[2] > M_PI)
      this->position[2] -= 2.0*M_PI;
    else if(this->position[2] < -M_PI)
      this->position[2] += 2.0*M_PI;
      
    // X/Y Velocity
    // Local coordinate frame: change of velocity in the last step
    dv[0] = ax * DT;
    dv[1] = ay * DT;

    // Global coordinate frame: velocity
    //   dependent on orientation.
    // Orientation in left-handed coordinate frame;
    //   Position in right-handed coordinate frame (need to fix this in future versions)
    float csgn = cos(this->position[2]);
    float ssgn = sin(this->position[2]);

    dvg[0] = csgn * dv[0] + ssgn * dv[1];
    dvg[1] = -ssgn * dv[0] + csgn * dv[1];
    this->velocity[0] += dvg[0];
    this->velocity[1] += dvg[1];   
    
    //if(this->debug_flag == 1)
    //  Serial.printf("DV: %3.7f %3.7f %3.7f %3.7f\n", dv[0], dv[1], dvg[0], dvg[1]);
      
    // X/Y Position: Global coordinate frame
    this->position[0] += this->velocity[0] * DT;
    this->position[1] += this->velocity[1] * DT;

    // Check for collision
    SegmentType collision = this->map.collision(this->position, RADIUS);
    if(collision != NONE)
    {
      // Collision: roll the position back to last state & set velocity to zero
      this->position[0] -= this->velocity[0] * DT;
      this->position[1] -= this->velocity[1] * DT;
      /*
       * // Nice idea, but needs more work
      Serial.println("Collision");
      if(collision == HORIZONTAL)
        this->velocity[1] *= -.5;
      else
        this->velocity[0] *= -.5;
       */
       // Velocities to zero
      this->velocity[0] = this->velocity[1] = 0.0;
      this->collisions++;
    }else{
      // Also need local coordinate frame change of position
      this->accumulated_position[0] += (csgn * this->velocity[0] - ssgn * this->velocity[1]) * DT;
      this->accumulated_position[1] += (ssgn * this->velocity[0] + csgn * this->velocity[1]) * DT;
    }
    
    //if(this->debug_flag == 1)
    //  Serial.printf("Delta: %.06f %0.6f %0.6f\n", dx[0], dx[1], dx[2]);
    
  }else{
    // No central fan: velocity is zero
    for(int i = 0; i < N_DIM; ++i) {
      this->velocity[i] = 0.0;
    }
  }
  
}
