/*
 * Model of the environment map
 * 
 * NOTE: there is no error checking for the bounds of the map.
 * 
 * Author: Andrew H. Fagg   Symbiotic Computing Laboratory 2020-04-15
 * 
 */

#include <Arduino.h>

#include "Map.h"

Map::Map()
{
  // Build the map: a set of line segments
  this->segments.push_back(new Segment(HORIZONTAL, 0, 0.0, 4.0));
  this->segments.push_back(new Segment(HORIZONTAL, 3.0, 0.0, 4.0));
  this->segments.push_back(new Segment(VERTICAL, 0, 0.0, 3.0));
  this->segments.push_back(new Segment(VERTICAL, 4.0, 0.0, 3.0));
  this->segments.push_back(new Segment(VERTICAL, 1.0, 0.0, 2.0));
  this->segments.push_back(new Segment(VERTICAL, 2.0, 1.0, 3.0));
  this->segments.push_back(new Segment(VERTICAL, 3.0, 0.0, 2.0));

  // Construct the 2D ASCII representation of the map
  this->populate_map();
}

/**
 * Given the set of line segments, draw the 2D ASCII representation of the map.
 * 
 */
void Map::populate_map()
{
  // Clear map
  for(int r = 0; r < MAP_ROWS; ++r)
  {
    for(int c = 0; c < MAP_COLS; ++c) 
    {
      this->map[r][c] = ' ';
    }
    this->map[r][MAP_COLS] = '\0';
  }
  
  // Iterate over all segments in the map
  for(std::vector<Segment*>::iterator it = this->segments.begin(); it != this->segments.end(); ++it){
    //Serial.printf("SEG: %d\n", (*it)->type);
    //delay(100);
    if((*it)->type == HORIZONTAL)
    {
      // Horizontal case
      int row = (int) (M2RPIXEL * (*it)->x);
      int c0 = (int) (M2CPIXEL * (*it)->y[0]);
      int c1 = (int) (M2CPIXEL * (*it)->y[1]);

      for(int c = c0; c <= c1; ++c) {
        this->map[MAP_ROWS-row-1][MAP_COLS-1-c] = '-';
      }
      
    }else{
      // Vertical case
      int col = (int) (M2CPIXEL * (*it)->x);
      int r0 = (int) (M2RPIXEL * (*it)->y[0]);
      int r1 = (int) (M2RPIXEL * (*it)->y[1]);
      
      for(int r = r0; r <= r1; ++r) {
        this->map[MAP_ROWS-r-1][MAP_COLS-1-col] = '|';
      }
    }
  }
}

/**
 * Print the map
 * 
 * @param X Array of floats encoding the location of the houvercraft (x,y in m)
 * @param c Character to use to display the hovercraft
 */
void Map::print(float X[2], char c)
{
  int row = MAP_ROWS-1 - (int) (M2RPIXEL * X[0]);
  int col = MAP_COLS-1 - (int) (M2CPIXEL * X[1]);

  //Serial.printf("%d %d\n", row, col);
  
  // Remember what the original character was
  char save = this->map[row][col];

  // Place the marker
  this->map[row][col] = c;

  // Print the map
  for(int r = 0; r < MAP_ROWS; ++r)
  {
    Serial.println(this->map[r]);  
  }

  // Restore saved value
  this->map[row][col] = save;
}

/**
 * Ask whether the houvercraft has collided with one of the
 * walls in the map
 * 
 * @param X Array of floats encoding the location of the houvercraft (x,y in m)
 * @param radius Radius of the houvercraft in m
 */
SegmentType Map::collision(float X[2], float radius)
{
  // Iterate over all segments in the map
  for(std::vector<Segment*>::iterator it = this->segments.begin(); it != this->segments.end(); ++it){
    // Collision with this segment?
    SegmentType collision = (*it)->collision(X, radius);
    if(collision != NONE)
      return(collision);
  }
  // No collisions
  return(NONE);
}

/**
 * Compute the smallest distance at which a distance sensor ray intersects
 * with one of the line segments
 * 
 * @param X Array of floats encoding the location of the houvercraft (x,y in m)
 * @param theta Orientation of the sensor (radians in the global coordinate frame)
 * 
 * @return Distance to the nearest object in m (range is 0 to MAX_DISTANCE; the latter
 *    indicates that nothing has been detected.
 */
float Map::intersection_distance(float X[2], float theta){
  float distance = Segment::MAX_DISTANCE;
  
  // Iterate over all segments in the map
  for(std::vector<Segment*>::iterator it = this->segments.begin(); it != this->segments.end(); ++it){
    // Distance to this segment
    float d = (*it)->intersection_distance(X, theta);

    // Is this one closer?
    if(d < distance)
        distance = d;
  }
  // Done
  return(distance);
}
