/*
 * Model of a single line segment.
 * 
 * Line segments are either vertical or horizontal.  
 *
 * If vertical, then
 * the class variable x is used to indicate the horizontal position of the
 * line segment and y[0/1] is used to indicate the ending coordinates of the
 * segment.
 * 
 * If horizontal, then the class variable x is used to indicate the vertical position
 * of the line segmetn and y[0/1] to indicate the ending coordinates of the segment
 * 
 * 
 * 
 * Author: Andrew H. Fagg   Symbiotic Computing Laboratory 2020-04-15
 * 
 */
 #include "Segment.h"

#define MIN(x,y) (((y0) < (y1)) ? (y0) : (y1))
#define MAX(x,y) (((y0) > (y1)) ? (y0) : (y1))

/**
 * Constructor: create a line segment.
 */
Segment::Segment(SegmentType type, float x, float y0, float y1)
{
  this->type = type;
  this->x = x;
  this->y[0] = MIN(y0, y1);
  this->y[1] = MAX(y0, y1);
}

/**
 * Detect a collision between a circle and the segment
 * 
 * @param X Array of floats encoding the location of the houvercraft (x,y in m)
 * @param radius Radius of the circle (in m)
 * 
 */
SegmentType Segment::collision(float X[2], float radius)
{
  if(this->type == HORIZONTAL) {
    // Horizontal case
    if(fabs(X[0] - this->x) > radius)
      return(NONE);
    if(X[1] < this->y[0] || X[1] > this->y[1])
      return(NONE);
    return(HORIZONTAL);
  }else{
    // Vertical case
    if(fabs(X[1] - this->x) > radius)
      return(NONE);
    if(X[0] < this->y[0] || X[0] > this->y[1])
      return(NONE);
    return(VERTICAL);
  }
}

/**
 * Compute the distance along a ray rooted at a position with orientation theta
 *  to the segment
 *  
 * 
 * @param X Array of floats encoding the location of the houvercraft (x,y in m)
 * @param theta Orientation of the ray in the global coordinate frame (in radians)
 * 
 * @return distance to the line segment from X along orientation theta.  MAX_DISTANCE if
 *   the distance is too large or there is no intersection.
 * 
 */
float Segment::intersection_distance(float X[2], float theta){
  if(this->type == VERTICAL){
    float stheta = sin(theta);

    // Are we almost parallel to the wall?
    if(fabs(stheta) < .001)
      return MAX_DISTANCE;
      
    // Horizontal distance to wall (x is the fixed coord)
    float dy = X[1] - this->x; 
    float d = dy / stheta;

    // Pointed in the right direction?
    if(d < 0)
      return MAX_DISTANCE;

    // vertical distance
    float dx = dy / tan(theta);

    // Intersection height 
    float xwall = X[0] + dx;

    // Above the highest point?
    if(xwall > this->y[1])
      return MAX_DISTANCE;
    // Below the lowest point?
    if(xwall < this->y[0])
      return MAX_DISTANCE;

    // Sensor limit
    if(d > MAX_DISTANCE)
      return MAX_DISTANCE;

    return(d);
  }else{
    // Angle to the wall
    float ctheta = cos(theta);

    // If almost parallel, then return
    if(fabs(ctheta) < .001)
       return MAX_DISTANCE;

    // Change in positin forward
    float dx = X[0] - this->x;

    // Distance of ray intersection to the wall
    float d = - dx / ctheta;

    // If ray distance is backwards, the return
    if(d < 0)
       return MAX_DISTANCE;

    // Horizontal distance (relative) along wall
    float dy = dx * tan(theta);

    // Global position along wall
    float ywall = X[1] + dy;

    // Check wall boundaries
    if(ywall < this->y[0])
       return MAX_DISTANCE;
    if(ywall > this->y[1])
       return MAX_DISTANCE;

    // check against max distance
    if(d > MAX_DISTANCE)
       return(MAX_DISTANCE);

    // There is an object
    return(d);
  }
};
